<?php
/**
 * IP Country + Type Detector (Human vs Bot/Datacenter/CDN)
 * + Writes BOT/SERVER IPs into bots.txt automatically (unique, appended safely)
 *
 * Usage:
 * - Put ip_check.php and ip.txt in same folder (e.g., public_html/ip_check/)
 * - Open ip_check.php in browser, it will read ip.txt by default
 * - Any IP classified as BOT/SERVER will be written to bots.txt (one per line, unique)
 *
 * Data source: ip-api.com (no key needed)
 */

declare(strict_types=1);
header('Content-Type: text/html; charset=utf-8');

function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function extract_ips(string $raw): array {
    preg_match_all('/\b(?:(?:25[0-5]|2[0-4]\d|1?\d?\d)\.){3}(?:25[0-5]|2[0-4]\d|1?\d?\d)\b/', $raw, $m);
    $ips = array_unique($m[0] ?? []);
    sort($ips);
    return $ips;
}

function api_lookup_many(array $ips): array {
    if (!$ips) return [];
    $payload = [];
    foreach ($ips as $ip) $payload[] = ["query" => $ip];

    $ch = curl_init('http://ip-api.com/batch?fields=status,message,query,country,countryCode,regionName,city,isp,org,as,hosting,proxy,mobile');
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_TIMEOUT => 20,
    ]);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);

    if ($res === false) throw new RuntimeException("cURL error: " . $err);

    $data = json_decode($res, true);
    if (!is_array($data)) throw new RuntimeException("Bad JSON response from ip-api.com");

    $out = [];
    foreach ($data as $row) {
        if (!is_array($row)) continue;
        $ip = $row['query'] ?? '';
        if ($ip) $out[$ip] = $row;
    }
    return $out;
}

function classify(array $row): array {
    $org = strtolower(trim(($row['org'] ?? '') . ' ' . ($row['isp'] ?? '') . ' ' . ($row['as'] ?? '')));
    $botKeywords = [
        'cloudflare','fastly','google','googlebot','gcp','microsoft','azure',
        'amazon','aws','digitalocean','ovh','hetzner','linode','vultr',
        'akamai','edgecast','stackpath','cdn','datacenter','hosting','proxy'
    ];
    $isHosting = (bool)($row['hosting'] ?? false);
    $isProxy   = (bool)($row['proxy'] ?? false);
    $isMobile  = (bool)($row['mobile'] ?? false);

    $hit = false;
    foreach ($botKeywords as $kw) {
        if (str_contains($org, $kw)) { $hit = true; break; }
    }

    if ($isProxy || $isHosting) return ['BOT/SERVER (High)', 'hosting/proxy=true'];
    if ($hit && !$isMobile)     return ['BOT/SERVER (Medium)', 'Org/ASN matches CDN/Cloud keywords'];
    if ($isMobile)              return ['HUMAN (High)', 'Mobile network'];
    return ['HUMAN (Medium)', 'Residential/ISP likely'];
}

function load_existing_bot_ips(string $path): array {
    if (!is_file($path)) return [];
    $raw = file_get_contents($path);
    if ($raw === false) return [];
    $ips = extract_ips($raw);
    $set = [];
    foreach ($ips as $ip) $set[$ip] = true;
    return $set;
}

function save_bot_ips_unique(string $path, array $botIps): void {
    // Ensure folder is writable.
    $existing = load_existing_bot_ips($path);
    foreach ($botIps as $ip) $existing[$ip] = true;

    $final = array_keys($existing);
    sort($final);
    $data = implode("\n", $final) . (count($final) ? "\n" : "");

    // Atomic write
    $tmp = $path . ".tmp";
    file_put_contents($tmp, $data, LOCK_EX);
    rename($tmp, $path);
}

// ---------- Input ----------
$inputText = '';
$source = 'ip.txt';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $inputText = (string)($_POST['ips'] ?? '');
    $source = 'manual input';
} else {
    $file = __DIR__ . '/ip.txt';
    if (is_file($file)) $inputText = (string)file_get_contents($file);
}

$ips = extract_ips($inputText);
$rows = [];
$error = null;
$botIpsDetected = [];

try {
    $lookup = api_lookup_many($ips);
    foreach ($ips as $ip) {
        $row = $lookup[$ip] ?? ['status' => 'fail', 'message' => 'not found', 'query' => $ip];
        $rows[] = $row;

        if (($row['status'] ?? '') === 'success') {
            [$type, $why] = classify($row);
            if (str_starts_with($type, 'BOT')) $botIpsDetected[] = $ip;
        }
    }

    // Write bot IPs to bots.txt
    $botsFile = __DIR__ . '/bots.txt';
    save_bot_ips_unique($botsFile, $botIpsDetected);

} catch (Throwable $e) {
    $error = $e->getMessage();
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>IP Country + Bot Detector</title>
  <style>
    body { font-family: Arial, sans-serif; margin: 18px; }
    textarea { width: 100%; height: 140px; }
    table { border-collapse: collapse; width: 100%; margin-top: 14px; }
    th, td { border: 1px solid #ddd; padding: 8px; font-size: 14px; vertical-align: top; }
    th { background: #f5f5f5; }
    .tag { padding: 3px 8px; border-radius: 10px; display: inline-block; font-weight: bold; }
    .bot { background:#ffe1e1; }
    .human { background:#e6ffea; }
    .muted { color:#777; font-size: 13px; }
    code { background:#f2f2f2; padding:2px 6px; border-radius:6px; }
  </style>
</head>
<body>
  <h2>فحص IP: الدولة + هل بوت أم حقيقي</h2>

  <p class="muted">
    أي IP يتم تصنيفه <b>BOT/SERVER</b> سيتم حفظه تلقائيًا في الملف:
    <code>bots.txt</code> (داخل نفس المجلد) — بشكل <b>Unique</b> بدون تكرار.
  </p>

  <form method="post">
    <div class="muted">المصدر الحالي: <b><?=h($source)?></b> — تقدر تلصق IPات هنا بدل الملف.</div>
    <textarea name="ips" placeholder="الصق IPات هنا، كل IP في سطر..."><?=h($inputText)?></textarea>
    <button type="submit">فحص</button>
  </form>

  <?php if ($error): ?>
    <p style="color:red;">خطأ: <?=h($error)?></p>
  <?php endif; ?>

  <p class="muted">عدد IPات: <b><?=count($ips)?></b> — بوتات مكتشفة في هذه العملية: <b><?=count($botIpsDetected)?></b></p>

  <table>
    <thead>
      <tr>
        <th>IP</th>
        <th>الدولة</th>
        <th>المدينة/المنطقة</th>
        <th>ISP / ORG / ASN</th>
        <th>النوع</th>
        <th>سبب التصنيف</th>
      </tr>
    </thead>
    <tbody>
    <?php foreach ($rows as $r): ?>
      <?php
        $ip = $r['query'] ?? '';
        if (($r['status'] ?? '') !== 'success') {
            $country = 'Unknown';
            $place = '-';
            $isp = '-';
            $type = 'Unknown';
            $why = $r['message'] ?? 'fail';
            $cls = '';
        } else {
            $country = ($r['country'] ?? 'Unknown') . ' (' . ($r['countryCode'] ?? '-') . ')';
            $place = trim(($r['regionName'] ?? '') . ' / ' . ($r['city'] ?? ''));
            $isp = trim(($r['isp'] ?? '') . ' | ' . ($r['org'] ?? '') . ' | ' . ($r['as'] ?? ''));
            [$type, $why] = classify($r);
            $cls = str_starts_with($type, 'BOT') ? 'bot' : 'human';
        }
      ?>
      <tr>
        <td><?=h($ip)?></td>
        <td><?=h($country)?></td>
        <td><?=h($place ?: '-')?></td>
        <td><?=h($isp)?></td>
        <td><span class="tag <?=h($cls)?>"><?=h($type)?></span></td>
        <td><?=h($why)?></td>
      </tr>
    <?php endforeach; ?>
    </tbody>
  </table>

  <p class="muted">
    لو ملف <code>bots.txt</code> ما يتكتبش: تأكد صلاحيات المجلد (writable) على الاستضافة.
  </p>
</body>
</html>
